/**
* \file: msd_event.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Meera Baby / RBEI / SWGII / meera.baby@in.bosch.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <libudev.h>
#include <blkid/blkid.h>

#include "backends/msd_events.h"
#include "backends/msd-metadata.h"
#include "utils/logger.h"

#define BLKID_PROBE_FREE NULL
#define DEV_NODE_NOT_IN_CACHE NULL
#define DEV_TYPE_NOT_IN_CACHE NULL
#define FILESYSTEM_NOT_IN_CACHE NULL
#define FILESYSTEM_UUID_NOT_IN_CACHE NULL
#define PARTITION_COUNT_NOT_IN_CACHE -1
#define PARTITION_NO_NOT_IN_CACHE -1
#define CDDA_NOT_IN_CACHE 0

typedef struct msd_uevent_info_cache
{
	struct udev_device *udevice_uevent;
	blkid_probe pr;
	const char *dev_node;
	const char *dev_type;
	const char *file_system;
	const char *filesystem_uuid;
	int partition_cnt;
	int partition_number;
	bool contains_cdda;
}msd_uevent_info_cache_t;

/**
 * Cache functions to extract the information from uevent/sysfs on first access to  event info cache structure
 */

static void msd_uevent_cache_dev_node(msd_uevent_info_cache_t *uevent_info_cache);

static void msd_uevent_cache_dev_type(msd_uevent_info_cache_t *uevent_info_cache);

static bool msd_uevent_has_prefix(const char *prefix, const char *child);

static int msd_uevent_manual_check_partition(struct udev_device *udevice);

static int msd_uevent_blkid_probe(const char *dev_node, blkid_probe *pr, int *fd);

static int msd_uevent_blkid_part_count(msd_uevent_info_cache_t *uevent_info_cache);

static void msd_uevent_cache_partition_cnt(msd_uevent_info_cache_t *uevent_info_cache);

static void msd_uevent_blkid_enable_probe_flags(msd_uevent_info_cache_t *uevent_info_cache);

static void msd_uevent_blkid_lookup(msd_uevent_info_cache_t *uevent_info_cache);

static void msd_uevent_blkid_probe_generic(msd_uevent_info_cache_t *uevent_info_cache);

static void msd_uevent_cache_partition_number(msd_uevent_info_cache_t *uevent_info_cache);

static void msd_uevent_cache_filesystem(msd_uevent_info_cache_t *uevent_info_cache);

static void msd_uevent_cache_filesystem_uuid(msd_uevent_info_cache_t *uevent_info_cache);

static void msd_uevent_cache_contains_cdda(msd_uevent_info_cache_t *uevent_info_cache);

static bool msd_uevent_cd_contains_data_track(msd_uevent_info_cache_t *uevent_info_cache);

static bool msd_uevent_is_cd_drive(msd_uevent_info_cache_t *uevent_info_cache);

//-------------------------------- Uevent info cache member initialization ----------------------------------------------------
void msd_uevent_init(struct udev_device *udevice, msd_uevent_info_cache_t *uevent_info_cache)
{
	uevent_info_cache->udevice_uevent=udev_device_ref(udevice);
	uevent_info_cache->pr=BLKID_PROBE_FREE;
	uevent_info_cache->dev_node=DEV_NODE_NOT_IN_CACHE;
	uevent_info_cache->dev_type=DEV_TYPE_NOT_IN_CACHE;
	uevent_info_cache->file_system=FILESYSTEM_NOT_IN_CACHE;
	uevent_info_cache->filesystem_uuid=FILESYSTEM_UUID_NOT_IN_CACHE;
	uevent_info_cache->partition_cnt=PARTITION_COUNT_NOT_IN_CACHE;
	uevent_info_cache->partition_number=PARTITION_NO_NOT_IN_CACHE;
	uevent_info_cache->contains_cdda=CDDA_NOT_IN_CACHE;
	return;
}

//-------------------------------- Uevent info cache member deinitialization ----------------------------------------------------
void msd_uevent_deinit(msd_uevent_info_cache_t *uevent_info_cache)
{
	udev_device_unref(uevent_info_cache->udevice_uevent);
	uevent_info_cache->udevice_uevent=NULL;
	if(uevent_info_cache->pr!=BLKID_PROBE_FREE)
		blkid_free_probe(uevent_info_cache->pr);
	uevent_info_cache->pr=BLKID_PROBE_FREE;
	uevent_info_cache->dev_node=DEV_NODE_NOT_IN_CACHE;
	uevent_info_cache->dev_type=DEV_TYPE_NOT_IN_CACHE;
	uevent_info_cache->file_system=FILESYSTEM_NOT_IN_CACHE;
	uevent_info_cache->filesystem_uuid=FILESYSTEM_UUID_NOT_IN_CACHE;
	uevent_info_cache->partition_cnt=PARTITION_COUNT_NOT_IN_CACHE;
	uevent_info_cache->partition_number=PARTITION_NO_NOT_IN_CACHE;
	uevent_info_cache->contains_cdda=CDDA_NOT_IN_CACHE;
	return;
}

//--------------------------------  Uevent info cache: cache functions  ----------------------------------------------------
static void msd_uevent_cache_dev_node(msd_uevent_info_cache_t *uevent_info_cache)
{
	assert(uevent_info_cache->udevice_uevent!=NULL);
	uevent_info_cache->dev_node=udev_device_get_devnode(uevent_info_cache->udevice_uevent);
}

static void msd_uevent_cache_dev_type(msd_uevent_info_cache_t *uevent_info_cache)
{
	assert(uevent_info_cache->udevice_uevent!=NULL);
	uevent_info_cache->dev_type=udev_device_get_devtype(uevent_info_cache->udevice_uevent);
}

static bool msd_uevent_has_prefix(const char *prefix, const char *child)
{
	size_t lenpre = strlen(prefix);
	size_t lenchi = strlen(child);
	return ((lenchi > lenpre) ? (strncmp(prefix, child, lenpre) == 0) : 0);
}

static int msd_uevent_manual_check_partition(struct udev_device *udevice)
{
	struct udev *udev = NULL;
	struct udev_enumerate *udev_enumerate = NULL;
	struct udev_list_entry *item = NULL, *first = NULL;
	error_code_t result = RESULT_OK;
	int partition_count = 0;
	const char * prefix, *child;

	prefix = udev_device_get_syspath(udevice);
	if (!prefix)
	{
		result = RESULT_INVALID;
	}

	if (result == RESULT_OK)
	{
		udev = udev_device_get_udev(udevice);
		if (!udev)
			result = RESULT_INVALID;
	}

	if (result == RESULT_OK)
	{
		udev_enumerate = udev_enumerate_new(udev);
		if (!udev_enumerate)
			result = RESULT_INVALID;
	}

	if (result == RESULT_OK)
	{
		if (udev_enumerate_add_match_parent(udev_enumerate, udevice) < 0)
			result = RESULT_INVALID;
	}

	if (result == RESULT_OK)
	{
		if (udev_enumerate_scan_devices(udev_enumerate) < 0)
			result = RESULT_INVALID;
	}

	if (result == RESULT_OK)
	{
		first = udev_enumerate_get_list_entry(udev_enumerate);
		udev_list_entry_foreach(item, first)
		{
			child = udev_list_entry_get_name(item);
			if (msd_uevent_has_prefix(prefix, child))
			{
				partition_count++;
			}
		}
	}
	/* Free the enumerator object */
	udev_enumerate_unref(udev_enumerate);
	return partition_count;
}

 static int msd_uevent_blkid_probe(const char *dev_node, blkid_probe *pr, int *fd)
{
	int err = -1;

	/*Start a new probe*/
	*pr = blkid_new_probe();
	if (*pr)
	{
		/*Get the file descriptor of the device file*/
		*fd = open(dev_node, O_RDONLY | O_CLOEXEC);
		if ((*fd) >= 0)
			/*Assigns the whole device from the beginning to probe control struct*/
			err = blkid_probe_set_device(*pr, *fd, 0, 0);
	}
	return err;
}

static int msd_uevent_blkid_part_count(msd_uevent_info_cache_t *uevent_info_cache)
{
	blkid_partlist ls;
	int nparts = -1;
	int err;
	int fd = -1;

	err = msd_uevent_blkid_probe(msd_uevent_get_dev_node(uevent_info_cache),
			&(uevent_info_cache->pr), &fd);
	if (!err)
	{
		ls = blkid_probe_get_partitions(uevent_info_cache->pr);
		nparts = blkid_partlist_numof_partitions(ls);
		close(fd);
	}
	if(uevent_info_cache->pr)
		blkid_free_probe(uevent_info_cache->pr);
	uevent_info_cache->pr = BLKID_PROBE_FREE;

	return nparts;
}

static void msd_uevent_cache_partition_cnt(msd_uevent_info_cache_t *uevent_info_cache)
{
	int partition_cnt;
	assert(uevent_info_cache->udevice_uevent!=NULL);
	partition_cnt=msd_uevent_get_intvalue_from_udevice_property(uevent_info_cache, "UDISKS_PARTITION_TABLE_COUNT", -1);
	if(partition_cnt == -1)
		partition_cnt=msd_uevent_blkid_part_count(uevent_info_cache);
	if(partition_cnt == -1)
		partition_cnt=msd_uevent_manual_check_partition(uevent_info_cache->udevice_uevent);
	uevent_info_cache->partition_cnt=partition_cnt;
	return;
}

static void msd_uevent_blkid_enable_probe_flags(msd_uevent_info_cache_t *uevent_info_cache)
{
	int superblock_probe_flag = 0;
	int partition_probe_flag = 0;
	bool do_superblock_probe = 0;
	bool do_partition_probe = 0;

	if(!uevent_info_cache->file_system) 
	{
		do_superblock_probe = 1;
		superblock_probe_flag |= BLKID_SUBLKS_TYPE;
	}
	if(!uevent_info_cache->filesystem_uuid)
	{
		do_superblock_probe = 1;
		superblock_probe_flag |= BLKID_SUBLKS_UUID;
	}
	if((uevent_info_cache->partition_number == -1) && 
			(!strcmp(msd_uevent_get_dev_type(uevent_info_cache),"partition")))
	{
			do_partition_probe = 1;
			partition_probe_flag |= BLKID_PARTS_ENTRY_DETAILS;
	}

	/* Enable probing and set probing flags to the prober */
	if (do_partition_probe) {
		blkid_probe_enable_partitions(uevent_info_cache->pr, 1);
		blkid_probe_set_partitions_flags(uevent_info_cache->pr,
				partition_probe_flag);
	}

	if (do_superblock_probe) {
		blkid_probe_enable_superblocks(uevent_info_cache->pr, 1);
		blkid_probe_set_superblocks_flags(uevent_info_cache->pr,
				superblock_probe_flag);
	}
	return;
}

static void msd_uevent_blkid_lookup(msd_uevent_info_cache_t *uevent_info_cache)
{
	const char *data_str = NULL;
	char *data=NULL;
	int part_value = -1;
	int err = 0;

	if (!uevent_info_cache->file_system)
	{
		err = blkid_probe_lookup_value(uevent_info_cache->pr, "TYPE", &data_str, NULL);
		if (err >= 0)
			uevent_info_cache->file_system = data_str;
	}

	if(!uevent_info_cache->filesystem_uuid)
	{
		err = blkid_probe_lookup_value(uevent_info_cache->pr, "UUID", &data_str, NULL);
		if (err >= 0)
			uevent_info_cache->filesystem_uuid = data_str;
	}

	if((uevent_info_cache->partition_number == -1) && 
			(!strcmp(msd_uevent_get_dev_type(uevent_info_cache),"partition")))
	{
		err = blkid_probe_lookup_value(uevent_info_cache->pr, "PART_ENTRY_NUMBER", &data_str, NULL);
		if ((err >= 0) && (data_str != NULL))
		{
			part_value = strtol(data_str, &data, 10);
			//unable to parse the number
			if (data_str == data)
				part_value = -1;
		}
		uevent_info_cache->partition_number = part_value;
	}
	return;
}

static void msd_uevent_blkid_probe_generic (msd_uevent_info_cache_t *uevent_info_cache)
{
	int err;
	int fd = -1;

	err = msd_uevent_blkid_probe(msd_uevent_get_dev_node(uevent_info_cache),
			&(uevent_info_cache->pr), &fd);

	if(!err)
	{
		msd_uevent_blkid_enable_probe_flags(uevent_info_cache);
		err = blkid_do_safeprobe(uevent_info_cache->pr);
		if(!err)
			msd_uevent_blkid_lookup(uevent_info_cache);
		close(fd);
	}

	return;
}

static void msd_uevent_cache_partition_number(msd_uevent_info_cache_t *uevent_info_cache)
{
	assert(uevent_info_cache->udevice_uevent!=NULL);
	uevent_info_cache->partition_number=msd_uevent_get_intvalue_from_udevice_property(uevent_info_cache, "UDISKS_PARTITION_NUMBER", -1);
	if(uevent_info_cache->partition_number == -1)
		msd_uevent_blkid_probe_generic(uevent_info_cache);
	//fallback in case no partion number can be determined.
	//This should happen only in cases we have no partition table at all. In this case, 1 is the correct value
	if(uevent_info_cache->partition_number == -1)
		uevent_info_cache->partition_number = 1;
}

static void msd_uevent_cache_filesystem(msd_uevent_info_cache_t *uevent_info_cache)
{
	assert(uevent_info_cache->udevice_uevent!=NULL);
	uevent_info_cache->file_system=udev_device_get_property_value(uevent_info_cache->udevice_uevent,"ID_FS_TYPE");
	if(!(uevent_info_cache->file_system))
		msd_uevent_blkid_probe_generic(uevent_info_cache);
}

static void msd_uevent_cache_filesystem_uuid(msd_uevent_info_cache_t *uevent_info_cache)
{
	assert(uevent_info_cache->udevice_uevent!=NULL);
	uevent_info_cache->filesystem_uuid=msd_uevent_get_value_from_udevice_property_or_default(uevent_info_cache,"ID_FS_UUID",NULL);
	if(!(uevent_info_cache->filesystem_uuid))
		msd_uevent_blkid_probe_generic(uevent_info_cache);
}

static void msd_uevent_cache_contains_cdda(msd_uevent_info_cache_t *uevent_info_cache)
{
	assert(uevent_info_cache->udevice_uevent!=NULL);
	uevent_info_cache->contains_cdda=(udev_device_get_property_value(uevent_info_cache->udevice_uevent, "ID_CDROM_MEDIA_TRACK_COUNT_AUDIO")!=NULL);
}

static bool msd_uevent_cd_contains_data_track(msd_uevent_info_cache_t *uevent_info_cache)
{
	assert(uevent_info_cache->udevice_uevent!=NULL);
	return udev_device_get_property_value(uevent_info_cache->udevice_uevent,"ID_CDROM_MEDIA_TRACK_COUNT_DATA")!=NULL;
}

static bool msd_uevent_is_cd_drive(msd_uevent_info_cache_t *uevent_info_cache)
{
	return udev_device_get_property_value(uevent_info_cache->udevice_uevent,"ID_CDROM")!=NULL;
}
//--------------------------------  Uevent info cache: get functions  ----------------------------------------------------
const char *msd_uevent_get_dev_node(msd_uevent_info_cache_t *uevent_info_cache)
{
	if(uevent_info_cache->dev_node==DEV_NODE_NOT_IN_CACHE)
		msd_uevent_cache_dev_node(uevent_info_cache);
	return uevent_info_cache->dev_node;
}

const char *msd_uevent_get_dev_type(msd_uevent_info_cache_t *uevent_info_cache)
{
	if(uevent_info_cache->dev_type==DEV_TYPE_NOT_IN_CACHE)
		msd_uevent_cache_dev_type(uevent_info_cache);
	return uevent_info_cache->dev_type;
}

int msd_uevent_get_partition_cnt(msd_uevent_info_cache_t *uevent_info_cache)
{
	//the following if statement is a merge of two special cases handled here:
	//------------------------------------------------------------------------

	// 1.) skip probing in case we have a cd without a data track. This prevents ugly error message in kernel logs when cdda only cds
	//     are in the drive (fixes SWGIII-15891)

	// 2.) iso9660 cdroms burned using Apples I-Tunes contain a mac partition table. This table is not used. The normal iso9660
	//     starts normally at the beginning of the data track and can be mounted directly.
	//     Automounter gets confused by this partition table because it indicates that two partitions will come up.
	//     He is waiting for the expected partitions which never will appear.
	//     The following if statements set the partition cnt to 0 in case we have detected a cd indicating
	//     that we have no partition table on the medium (SWGIII-15967)

	//The merge just checks if this event comes from a CD drive. If so, we just manually set the cnt of partitions to 0.
	if (msd_uevent_is_cd_drive(uevent_info_cache))
	{
		uevent_info_cache->partition_cnt = 0;
		return uevent_info_cache->partition_cnt;
	}

	if(uevent_info_cache->partition_cnt==PARTITION_COUNT_NOT_IN_CACHE)
		msd_uevent_cache_partition_cnt(uevent_info_cache);
	return uevent_info_cache->partition_cnt;
}

int msd_uevent_get_partition_number(msd_uevent_info_cache_t *uevent_info_cache)
{
	//skip probing in case we have a cd without a data track. This prevents ugly error message in kernel logs when cdda only cds
	//are in the drive
	if (msd_uevent_is_cd_drive(uevent_info_cache) && !msd_uevent_cd_contains_data_track(uevent_info_cache)) {
		uevent_info_cache->partition_number = 0;
		return uevent_info_cache->partition_number;
	}

	if(uevent_info_cache->partition_number==PARTITION_NO_NOT_IN_CACHE)
		msd_uevent_cache_partition_number(uevent_info_cache);
	return uevent_info_cache->partition_number;
}

const char *msd_uevent_get_filesystem(msd_uevent_info_cache_t *uevent_info_cache)
{
	//skip probing in case we have a cd without a data track. This prevents ugly error message in kernel logs when cdda only cds
	//are in the drive
	if (msd_uevent_is_cd_drive(uevent_info_cache) && !msd_uevent_cd_contains_data_track(uevent_info_cache)) {
		uevent_info_cache->file_system = NULL;
		return uevent_info_cache->file_system;
	}

	if(uevent_info_cache->file_system==FILESYSTEM_NOT_IN_CACHE)
		msd_uevent_cache_filesystem(uevent_info_cache);
	return uevent_info_cache->file_system;
}

const char *msd_uevent_get_filesystem_uuid(msd_uevent_info_cache_t *uevent_info_cache)
{
	//skip probing in case we have a cd without a data track. This prevents ugly error message in kernel logs when cdda only cds
	//are in the drive
	if (msd_uevent_is_cd_drive(uevent_info_cache) && !msd_uevent_cd_contains_data_track(uevent_info_cache)) {
		uevent_info_cache->file_system = NULL;
		return uevent_info_cache->file_system;
	}

	if(uevent_info_cache->filesystem_uuid==FILESYSTEM_UUID_NOT_IN_CACHE)
		msd_uevent_cache_filesystem_uuid(uevent_info_cache);
	return uevent_info_cache->filesystem_uuid;
}

bool msd_uevent_contains_cdda(msd_uevent_info_cache_t *uevent_info_cache)
{
	if(uevent_info_cache->contains_cdda==CDDA_NOT_IN_CACHE)
		msd_uevent_cache_contains_cdda(uevent_info_cache);
	return uevent_info_cache->contains_cdda;
}
//--------------------------------  Generic get uevent property functions  ----------------------------------------------------

const char *msd_uevent_get_value_from_udevice_property_or_default(msd_uevent_info_cache_t *uevent_info_cache, const char *property_name, const char *default_val)
{
	return msd_metadata_get_value_from_udevice_property_or_default(uevent_info_cache->udevice_uevent, property_name, default_val);
}

int msd_uevent_get_intvalue_from_udevice_property(msd_uevent_info_cache_t *uevent_info_cache, const char *property_name, int default_val)
{
	return msd_metadata_get_intvalue_from_udevice_property(uevent_info_cache->udevice_uevent, property_name, default_val);
}

//--------------------------------  Generic get sys attribute value function  ----------------------------------------------------

long msd_uevent_get_longvalue_sys_attr_or_default(msd_uevent_info_cache_t *uevent_info_cache, const char *property_name, int default_val)
{
	long longval=default_val;

	const char *value_str;
	char *value_str_ret;

	value_str=udev_device_get_sysattr_value(uevent_info_cache->udevice_uevent, property_name);
	if (value_str!=NULL)
	{
		longval=strtol(value_str,&value_str_ret,10);
		//unable to parse the number
		if (value_str==value_str_ret)
			longval=default_val;
	}

	return longval;
}
//--------------------------------  Function to get size of cache structure  ----------------------------------------------------
size_t msd_uevent_get_size_cache()
{
	return sizeof(msd_uevent_info_cache_t);
}

//--------------------------------  Determine parent wrapper function  ----------------------------------------------------
struct udev_device *msd_uevent_determine_parent_device(msd_uevent_info_cache_t *uevent_info_cache)
{
	const char *dev_type;

	struct udev_device *udevice;
	udevice=udev_device_get_parent(uevent_info_cache->udevice_uevent);

	if (udevice!=NULL)
	{
		dev_type=udev_device_get_devtype(udevice);
		if (dev_type!=NULL)
		{
			if (strcmp(dev_type,"disk")!=0)
				udevice=NULL;
		}
		else
			udevice=NULL;
	}
	return udevice;
}

